/*
 * sau3517.c  -- ALSA SoC support for OMAP3517 / AM3517 EVM
 *
 * Author: Anuj Aggarwal <anuj.aggarwal@ti.com>
 *
 * Based on sound/soc/omap/beagle.c by Steve Sakoman
 *
 * Copyright (C) 2009 Texas Instruments Incorporated
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any kind,
 * whether express or implied; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include <linux/clk.h>
#include <linux/platform_device.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>

#include <asm/mach-types.h>
#include <mach/hardware.h>
#include <mach/gpio.h>
#include <plat/mcbsp.h>

#include "omap-mcbsp.h"
#include "omap-pcm.h"

#include "../codecs/tlv320aic3x.h"

#define CODEC_CLOCK 	26000000

static int sau3517_codec_hw_params(struct snd_pcm_substream *substream,
	struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	int ret;

	/* Set codec DAI configuration */
	ret = snd_soc_dai_set_fmt(codec_dai,
				  SND_SOC_DAIFMT_I2S |
				  SND_SOC_DAIFMT_NB_NF |
				  SND_SOC_DAIFMT_CBM_CFM);
	if (ret < 0) {
		printk(KERN_ERR "can't set codec DAI configuration\n");
		return ret;
	}

	/* Set cpu DAI configuration */
	ret = snd_soc_dai_set_fmt(cpu_dai,
				  SND_SOC_DAIFMT_I2S |
				  SND_SOC_DAIFMT_NB_NF |
				  SND_SOC_DAIFMT_CBM_CFM);
	if (ret < 0) {
		printk(KERN_ERR "can't set cpu DAI configuration\n");
		return ret;
	}

	/* Set the codec system clock for DAC and ADC */
	ret = snd_soc_dai_set_sysclk(codec_dai, 0,
			CODEC_CLOCK, SND_SOC_CLOCK_IN);
	if (ret < 0) {
		printk(KERN_ERR "can't set codec system clock\n");
		return ret;
	}

	return 0;
}

static int sau3517_bt_hw_params(struct snd_pcm_substream *substream,
	struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	int ret;

	/* Set cpu DAI configuration */
	ret = snd_soc_dai_set_fmt(cpu_dai,
				  SND_SOC_DAIFMT_DSP_A |
				  SND_SOC_DAIFMT_NB_NF |
				  SND_SOC_DAIFMT_CBM_CFM);
	if (ret < 0) {
		printk(KERN_ERR "bt: can't set cpu DAI configuration\n");
		return ret;
	}

	return 0;
}

static struct snd_soc_ops sau3517_codec_ops = {
	.hw_params = sau3517_codec_hw_params,
};

static struct snd_soc_ops sau3517_bt_ops = {
	.hw_params = sau3517_bt_hw_params,
};

/* sau3517 machine dapm widgets */
static const struct snd_soc_dapm_widget tlv320aic3x_dapm_widgets[] = {
	SND_SOC_DAPM_VMID("HP VMID"),
	SND_SOC_DAPM_HP("Right HP Out", NULL),
	SND_SOC_DAPM_HP("Left HP Out", NULL),
	SND_SOC_DAPM_LINE("Right Line Out", NULL),
	SND_SOC_DAPM_LINE("Left Line Out", NULL),
	SND_SOC_DAPM_LINE("Right Line In 1", NULL),
	SND_SOC_DAPM_LINE("Left Line In 1", NULL),
	SND_SOC_DAPM_LINE("Right Line In 2", NULL),
	SND_SOC_DAPM_LINE("Left Line In 2", NULL),
	SND_SOC_DAPM_MIC("Right Mic In", NULL),
	SND_SOC_DAPM_MIC("Left Mic In", NULL),
};

static const struct snd_soc_dapm_route audio_map[] = {
	/* Line Out connected to LLOUT, RLOUT */

	{"HP VMID", NULL, "HPLCOM"},
	{"HP VMID", NULL, "HPRCOM"},

	{"Left Line Out", NULL, "LLOUT"},
	{"Right Line Out", NULL, "RLOUT"},

	{"Left HP Out", NULL, "HPLOUT"},
	{"Right HP Out", NULL, "HPROUT"},

	{"LINE1L", NULL, "Left Line In 1"},
	{"LINE1R", NULL, "Right Line In 1"},

	{"LINE2L", NULL, "Left Line In 2"},
	{"LINE2R", NULL, "Right Line In 2"},

	{"MIC3L", NULL, "Left Mic In"},
	{"MIC3R", NULL, "Right Mic In"},
};

static int sau3517_aic3x_init(struct snd_soc_pcm_runtime *rtd)
{
	struct snd_soc_codec *codec = rtd->codec;

	/* Add am3517-evm specific widgets */
	snd_soc_dapm_new_controls(codec, tlv320aic3x_dapm_widgets,
				  ARRAY_SIZE(tlv320aic3x_dapm_widgets));

	/* Set up davinci-evm specific audio path audio_map */
	snd_soc_dapm_add_routes(codec, audio_map, ARRAY_SIZE(audio_map));

	/* always connected */
	snd_soc_dapm_enable_pin(codec, "HP VMID");
	snd_soc_dapm_enable_pin(codec, "Left Line Out");
	snd_soc_dapm_enable_pin(codec, "Right Line Out");
	snd_soc_dapm_enable_pin(codec, "Left HP Out");
	snd_soc_dapm_enable_pin(codec, "Right HP Out");
	snd_soc_dapm_enable_pin(codec, "Left Line In 1");
	snd_soc_dapm_enable_pin(codec, "Right Line In 1");
	snd_soc_dapm_enable_pin(codec, "Left Line In 2");
	snd_soc_dapm_enable_pin(codec, "Right Line In 2");
	snd_soc_dapm_enable_pin(codec, "Left Mic In");
	snd_soc_dapm_enable_pin(codec, "Right Mic In");

	snd_soc_dapm_sync(codec);

	pr_info("AM3517/05 SAUModule dapm init\n");

	return 0;
}

/* Digital audio interface glue */
static struct snd_soc_dai_link sau3517_dai [] = {
	{
		.name = "TLV320AIC3X",
		.stream_name = "AIC3X",
		.cpu_dai_name ="omap-mcbsp-dai.1",
		.codec_dai_name = "tlv320aic3x-hifi",
		.platform_name = "omap-pcm-audio",
		.codec_name = "tlv320aic3x-codec.1-0018",
		.init = sau3517_aic3x_init,
		.ops = &sau3517_codec_ops,
	}, {
		.name = "BT-88W8688",
		.stream_name = "BT8688",
		.cpu_dai_name ="omap-mcbsp-dai.2",
		.codec_dai_name = "bt-88w8688",
		.platform_name = "omap-pcm-audio",
		.codec_name = "bt-88w8688-codec",
		.ops = &sau3517_bt_ops,
	}
};

/* Audio machine driver */
static struct snd_soc_card snd_soc_sau3517 = {
	.name = "Saumodule-AM3517",
	.dai_link = sau3517_dai,
	.num_links = ARRAY_SIZE(sau3517_dai),
};

static struct platform_device *sau3517_snd_device;

static int __init sau3517_soc_init(void)
{
	int ret;

	pr_info("AM3517/05 SAUModule sound system init\n");

	sau3517_snd_device = platform_device_alloc("soc-audio", -1);
	if (!sau3517_snd_device) {
		printk(KERN_ERR "Platform device allocation failed (snd)\n");
		return -ENOMEM;
	}

	platform_set_drvdata(sau3517_snd_device, &snd_soc_sau3517);

	ret = platform_device_add(sau3517_snd_device);
	if (ret)
		goto err1;

	return 0;

err1:
	printk(KERN_ERR "Unable to add snd platform device\n");
	platform_device_put(sau3517_snd_device);

	return ret;
}

static void __exit sau3517_soc_exit(void)
{
	platform_device_unregister(sau3517_snd_device);
}

module_init(sau3517_soc_init);
module_exit(sau3517_soc_exit);

MODULE_AUTHOR("Sergey Markov, SAURIS GmbH <sauris_support@scanti.ru>");
MODULE_DESCRIPTION("ALSA SoC AM3517/05 SAUModule");
MODULE_LICENSE("GPL v2");
