/*
 * LCD panel driver for Sharp LQ043T1DG01
 *
 * Copyright (C) 2009 Texas Instruments Inc
 * Author: Vaibhav Hiremath <hvaibhav@ti.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/err.h>

#include <plat/display.h>

static struct omap_video_timings techtoys_430_timings = {
	.x_res = 480,
	.y_res = 272,

	.pixel_clock	= 10164,

	.hsw		= 80,
	.hfp		= 10,
	.hbp		= 8,

	.vsw		= 14,
	.vfp		= 4,
	.vbp		= 3,
};

static int techtoys_430_panel_power_on(struct omap_dss_device *dssdev)
{
	int r;

	if (dssdev->state == OMAP_DSS_DISPLAY_ACTIVE)
		return 0;

	r = omapdss_dpi_display_enable(dssdev);
	if (r)
		goto err0;

	/* wait couple of vsyncs until enabling the LCD */
	msleep(50);

	if (dssdev->platform_enable) {
		r = dssdev->platform_enable(dssdev);
		if (r)
			goto err1;
	}

	return 0;
err1:
	omapdss_dpi_display_disable(dssdev);
err0:
	return r;
}


static void techtoys_430_panel_power_off(struct omap_dss_device *dssdev)
{
	if (dssdev->state != OMAP_DSS_DISPLAY_ACTIVE)
		return;

	if (dssdev->platform_disable)
		dssdev->platform_disable(dssdev);

	/* wait at least 5 vsyncs after disabling the LCD */
	msleep(100);

	omapdss_dpi_display_disable(dssdev);
}


static int techtoys_430_panel_probe(struct omap_dss_device *dssdev)
{

	dssdev->panel.config = OMAP_DSS_LCD_TFT | OMAP_DSS_LCD_IVS |
		OMAP_DSS_LCD_IHS;
	dssdev->panel.acb = 0x0;
	dssdev->panel.timings = techtoys_430_timings;

	return 0;
}

static void techtoys_430_panel_remove(struct omap_dss_device *dssdev)
{
}

static int techtoys_430_panel_enable(struct omap_dss_device *dssdev)
{
	int r;

	r =  techtoys_430_panel_power_on(dssdev);
	if (r)
		return r;

	dssdev->state = OMAP_DSS_DISPLAY_ACTIVE;

	return 0;
}

static void techtoys_430_panel_disable(struct omap_dss_device *dssdev)
{

	techtoys_430_panel_power_off(dssdev);
	dssdev->state = OMAP_DSS_DISPLAY_DISABLED;

}

static int techtoys_430_panel_suspend(struct omap_dss_device *dssdev)
{
	techtoys_430_panel_power_off(dssdev);
	dssdev->state = OMAP_DSS_DISPLAY_SUSPENDED;
	return 0;
}

static int techtoys_430_panel_resume(struct omap_dss_device *dssdev)
{
	int r;
	
	r =  techtoys_430_panel_power_on(dssdev);
	if (r)
		return r;

	dssdev->state = OMAP_DSS_DISPLAY_ACTIVE;

	return 0;
}

static struct omap_dss_driver techtoys_430_driver = {
	.probe		= techtoys_430_panel_probe,
	.remove		= techtoys_430_panel_remove,

	.enable		= techtoys_430_panel_enable,
	.disable	= techtoys_430_panel_disable,
	.suspend	= techtoys_430_panel_suspend,
	.resume		= techtoys_430_panel_resume,

	.driver         = {
		.name   = "techtoys_430_panel",
		.owner  = THIS_MODULE,
	},
};

static int __init techtoys_430_panel_drv_init(void)
{
	return omap_dss_register_driver(&techtoys_430_driver);
}

static void __exit techtoys_430_panel_drv_exit(void)
{
	omap_dss_unregister_driver(&techtoys_430_driver);
}

module_init(techtoys_430_panel_drv_init);
module_exit(techtoys_430_panel_drv_exit);
MODULE_LICENSE("GPL");
