/*
 *  TPS6507x Backlight Driver
 *
 *  Copyright (c) 2010 Sergey Markov
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/mutex.h>
#include <linux/fb.h>
#include <linux/backlight.h>
#include <linux/mfd/tps6507x.h>

/* Flag to signal when the battery is low */
#define TPS6507XBL_BATTLOW       BL_CORE_DRIVER1

static uint dimming_freq = 2;
static uint iset_in = 1;

module_param_named(dimming_freq, dimming_freq, uint, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(dimming_freq, "Dimming frequency: 0 - 100 Hz, 1 - 200 Hz, 2 - 500 Hz, 3 - 1000 Hz [Default: 2]");
module_param_named(iset_in, iset_in, uint, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(iset_in, "ISET input selector: 0 - ISET1, 1 - ISET2 [Default: 1]");

struct tps6507x_bl {
	struct backlight_device			*bldev;
	struct platform_device			*pdev;
};


static int tps6507x_read_u8(struct tps6507x_dev *dev, u8 reg, u8 *data)
{
  return dev->read_dev(dev,reg,1,data);
}

static int tps6507x_write_u8(struct tps6507x_dev *dev, u8 reg, u8 data)
{
  return dev->write_dev(dev,reg,1,&data);
}

static int tps6507xbl_send_intensity(struct backlight_device *bd)
{
	int intensity = bd->props.brightness;
	struct tps6507x_bl *bl = bl_get_data(bd);
	struct tps6507x_dev *dev = dev_get_drvdata(bl->pdev->dev.parent);

	if (bd->props.power != FB_BLANK_UNBLANK)
		intensity = 0;
	if (bd->props.fb_blank != FB_BLANK_UNBLANK)
		intensity = 0;

	if (bd->props.state & BL_CORE_FBBLANK)
		intensity = 0;
	if (bd->props.state & BL_CORE_SUSPENDED)
		intensity = 0;
	if (bd->props.state & TPS6507XBL_BATTLOW)
	    if (intensity > 20) 
		intensity = 20;

	if (intensity > 99) 
	    intensity = 99;

	if (!intensity) {
	  // switch sinks OFF
	  tps6507x_write_u8(dev, TPS6507X_REG_WLED_CTRL1, (dimming_freq & 3) << 4);
	} else {
	  // set intensity
	  tps6507x_write_u8(dev, TPS6507X_REG_WLED_CTRL2, (0x7F & intensity) | (((~iset_in) & 1) << 7));
	  // switch sinks ON
	  tps6507x_write_u8(dev, TPS6507X_REG_WLED_CTRL1, 0x80 | ((dimming_freq & 3) << 4));
	}

	return 0;
}

static int tps6507xbl_get_intensity(struct backlight_device *bd)
{
	struct tps6507x_bl *bl = bl_get_data(bd);
	struct tps6507x_dev *dev = dev_get_drvdata(bl->pdev->dev.parent);
	u8 reg;
  
	tps6507x_read_u8(dev, TPS6507X_REG_WLED_CTRL1, &reg);
	
	if ((reg & 0x80) == 0)
	    return 0;
	    
	tps6507x_read_u8(dev, TPS6507X_REG_WLED_CTRL2, &reg);
	
	return (int) (reg & 0x7F);
}

static struct backlight_ops tps6507xbl_ops = {
	.options = BL_CORE_SUSPENDRESUME,
	.get_brightness = tps6507xbl_get_intensity,
	.update_status  = tps6507xbl_send_intensity,
};

static int tps6507xbl_probe(struct platform_device *pdev)
{

	struct backlight_device *bd;
	struct backlight_properties props;
	struct tps6507x_bl *bl;

	bl = kzalloc(sizeof(struct tps6507x_bl), GFP_KERNEL);
	if (!bl)
		return -ENOMEM;

	bl->pdev = pdev;

	memset(&props, 0, sizeof(struct backlight_properties));
	props.max_brightness = 99;

	bd = backlight_device_register ("tps6507x-bl",
		&pdev->dev, bl, &tps6507xbl_ops, &props);

	if (IS_ERR (bd)) {
	    dev_err(&pdev->dev, "Failed to register driver\n");
	    kfree(bl);
	    return -EIO;
	}

	bl->bldev = bd;

	platform_set_drvdata(pdev, bl);

	bd->props.max_brightness = 99;
	bd->props.power = FB_BLANK_UNBLANK;
	bd->props.brightness = 30;
	backlight_update_status(bd);

	printk("TPS6507x Backlight Driver Initialized.\n");
	return 0;
}

static void tps6507xbl_shutdown(struct platform_device *pdev)
{
	struct tps6507x_bl *bl = platform_get_drvdata(pdev);
	struct backlight_device *bd = bl->bldev;

	printk("TPS6507x Backlight Driver shutdown\n");

	bd->props.power = 0;
	bd->props.brightness = 0;
	backlight_update_status(bd);

}

static int tps6507xbl_remove(struct platform_device *pdev)
{
	struct tps6507x_bl *bl = platform_get_drvdata(pdev);

	tps6507xbl_shutdown(pdev);
	backlight_device_unregister(bl->bldev);
	platform_set_drvdata(pdev, NULL);
	kfree(bl);

	printk("TPS6507x Backlight Driver removed\n");

	return 0;
}

static struct platform_driver tps6507xbl_driver = {
	.probe		= tps6507xbl_probe,
	.remove		= tps6507xbl_remove,
	.shutdown	= tps6507xbl_shutdown,
	.driver		= {
		.name	= "tps6507x-bl",
		.owner = THIS_MODULE,
	},
};

static int __init tps6507xbl_init(void)
{
	return platform_driver_register(&tps6507xbl_driver);
}

static void __exit tps6507xbl_exit(void)
{
	platform_driver_unregister(&tps6507xbl_driver);
}

module_init(tps6507xbl_init);
module_exit(tps6507xbl_exit);

MODULE_AUTHOR("Sergey Markov <sm@venus.ru>");
MODULE_DESCRIPTION("TPS6507x Backlight Driver");
MODULE_LICENSE("GPL");
