/*
 * otg.c -- USB OTG utility code
 *
 * Copyright (C) 2004 Texas Instruments
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/kernel.h>
#include <linux/device.h>

#include <linux/usb/otg.h>

static struct otg_transceiver *xceiv[2];

/**
 * otg_get_transceiver - find the (single) OTG transceiver
 *
 * Returns the transceiver driver, after getting a refcount to it; or
 * null if there is no such transceiver.  The caller is responsible for
 * calling otg_put_transceiver() to release that count.
 *
 * For use by USB host and peripheral drivers.
 */
struct otg_transceiver *otg_get_transceiver(int id)
{
	if (xceiv[id])
		get_device(xceiv[id]->dev);
	return xceiv[id];
}
EXPORT_SYMBOL(otg_get_transceiver);

/**
 * otg_put_transceiver - release the (single) OTG transceiver
 * @x: the transceiver returned by otg_get_transceiver()
 *
 * Releases a refcount the caller received from otg_get_transceiver().
 *
 * For use by USB host and peripheral drivers.
 */
void otg_put_transceiver(struct otg_transceiver *x)
{
	if (x)
		put_device(x->dev);
}
EXPORT_SYMBOL(otg_put_transceiver);

/**
 * otg_set_transceiver - declare the (single) OTG transceiver
 * @x: the USB OTG transceiver to be used; or NULL
 *
 * This call is exclusively for use by transceiver drivers, which
 * coordinate the activities of drivers for host and peripheral
 * controllers, and in some cases for VBUS current regulation.
 */
int otg_set_transceiver(struct otg_transceiver *x)
{
	if (x && xceiv[x->id])
		return -EBUSY;
	xceiv[x->id] = x;
	return 0;
}
EXPORT_SYMBOL(otg_set_transceiver);

/**
 * otg_set_transceiver - declare the (single) OTG transceiver
 * @x: the USB OTG transceiver to be used; or NULL
 *
 * This call is exclusively for use by transceiver drivers, which
 * coordinate the activities of drivers for host and peripheral
 * controllers, and in some cases for VBUS current regulation.
 */
int otg_reset_transceiver(struct otg_transceiver *x)
{
	if (x && xceiv[x->id])
		xceiv[x->id] = NULL;
	return 0;
}
EXPORT_SYMBOL(otg_reset_transceiver);
