/*
 * tps6507x.c  --  TPS6507x chip family multi-function driver
 *
 *  Copyright (c) 2010 RidgeRun (todd.fischer@ridgerun.com)
 *
 * Author: Todd Fischer
 *         todd.fischer@ridgerun.com
 *
 * Credits:
 *
 *    Using code from wm831x-*.c, wm8400-core, Wolfson Microelectronics PLC.
 *
 * For licencing details see kernel-base/COPYING
 *
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/i2c.h>
#include <linux/mfd/core.h>
#include <linux/mfd/tps6507x.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>

static struct mfd_cell tps6507x_devs[] = {
	{
		.name = "tps6507x-pmic",
	},
	{
		.name = "tps6507x-ts",
	},
	{
		.name = "tps6507x-bl",
	},
};

static int poweroff_enable = 1;
static int wakeup_by_power_button = 1;
static int wakeup_by_tsc = 1;

module_param_named(poweroff_enable, poweroff_enable, uint, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(poweroff_enable, "Enable poweroff event, initiated by power button");

module_param_named(wakeup_by_power_button, wakeup_by_power_button, uint, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(wakeup_by_power_button, "Enable wakeup on power button");

module_param_named(wakeup_by_tsc, wakeup_by_tsc, uint, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(wakeup_by_tsc, "Enable wakeup on power button");

static int tps6507x_i2c_read_device(struct tps6507x_dev *tps6507x, char reg,
				  int bytes, void *dest)
{
	struct i2c_client *i2c = tps6507x->i2c_client;
	struct i2c_msg xfer[2];
	int ret;

	/* Write register */
	xfer[0].addr = i2c->addr;
	xfer[0].flags = 0;
	xfer[0].len = 1;
	xfer[0].buf = &reg;

	/* Read data */
	xfer[1].addr = i2c->addr;
	xfer[1].flags = I2C_M_RD;
	xfer[1].len = bytes;
	xfer[1].buf = dest;

	ret = i2c_transfer(i2c->adapter, xfer, 2);
	if (ret == 2)
		ret = 0;
	else if (ret >= 0)
		ret = -EIO;

	return ret;
}

static int tps6507x_i2c_write_device(struct tps6507x_dev *tps6507x, char reg,
				   int bytes, void *src)
{
	struct i2c_client *i2c = tps6507x->i2c_client;
	/* we add 1 byte for device register */
	u8 msg[TPS6507X_MAX_REGISTER + 1];
	int ret;

	if (bytes > TPS6507X_MAX_REGISTER)
		return -EINVAL;

	msg[0] = reg;
	memcpy(&msg[1], src, bytes);

	ret = i2c_master_send(i2c, msg, bytes + 1);
	if (ret < 0)
		return ret;
	if (ret != bytes + 1)
		return -EIO;
	return 0;
}

static irqreturn_t tps6507x_isr(int irq, void *dev_id)
{
	struct i2c_client *i2c = dev_id;
	struct tps6507x_dev *tps6507x = i2c_get_clientdata(i2c);
	int ret;

	disable_irq_nosync(irq);

	ret = schedule_delayed_work(&tps6507x->isr_work,0);

	return IRQ_HANDLED;
}

static void tps6507x_isr_work_func(struct work_struct *workstruct)
{

	int ret;
	u8 irq_mask, new_mask;
	int button_state;

	struct delayed_work *delay_work =
		container_of(workstruct, struct delayed_work, work);
	struct tps6507x_dev *tps6507x =
		container_of(delay_work, struct tps6507x_dev, isr_work);

	// Read IRQ register (clear pending interrupts)
	irq_mask = 0;
	ret = tps6507x->read_dev(tps6507x, TPS6507X_REG_INT, 1, &irq_mask);
	if (ret != 0) {
		printk( "TPS6507x: sint: error reading IRQ mask\n");
		return;
	}

	if (irq_mask & 0x04) {  // PB_IN pulled low
		if (!tps6507x->skip_pbtn && tps6507x->pbtn_gpio >= 0) {
			// increment "pulldown" counter
			// to initiale GPIO monitoring
			tps6507x->pbtn_counter++;
		}
		tps6507x->skip_pbtn = 0;
	}

	if (irq_mask & 0x8) { // TSC int
		new_mask = 0x20;
		ret = tps6507x->write_dev(tps6507x, TPS6507X_REG_INT, 1, &new_mask);
		if (ret != 0) {
			printk( "TPS6507x: sint: error writing irq mask \n");
		}
		tps6507x->skip_pbtn = 0;
	}

	// handle power button GPIO
	if (tps6507x->pbtn_counter) {
		button_state = gpio_get_value(tps6507x->pbtn_gpio);
		if (button_state) {
			// power button released
			if (poweroff_enable && tps6507x->pbtn_counter > 3 && tps6507x->shutdown)
				(tps6507x->shutdown)();
			tps6507x->pbtn_counter = 0;
		} else {
			// power button pressed
			tps6507x->pbtn_counter++;
			if (poweroff_enable && tps6507x->pbtn_counter > (5000/20) && tps6507x->switch_off)
				(tps6507x->switch_off)();
		}
	}

	// Reenable IRQ or reschedule this work
	if ((irq_mask & 0x0F) || tps6507x->pbtn_counter) {
		schedule_delayed_work(&tps6507x->isr_work, msecs_to_jiffies(20));
	} else {
		enable_irq(tps6507x->i2c_client->irq);
	}

}

static int tps6507x_i2c_probe(struct i2c_client *i2c,
			    const struct i2c_device_id *id)
{
	struct tps6507x_board *pdata;
	struct tps6507x_dev *tps6507x;
	int ret = 0;
	u8 irq_mask;

	tps6507x = kzalloc(sizeof(struct tps6507x_dev), GFP_KERNEL);
	if (tps6507x == NULL)
		return -ENOMEM;

	pdata = i2c->dev.platform_data;

	if (pdata) {
		tps6507x->pbtn_gpio = pdata->pbtn_gpio;
		tps6507x->switch_off = pdata->switch_off;
		tps6507x->shutdown = pdata->shutdown;
	} else {
		tps6507x->pbtn_gpio = -EINVAL;
		tps6507x->switch_off = 
		tps6507x->shutdown = NULL;
	}

	i2c_set_clientdata(i2c, tps6507x);
	tps6507x->dev = &i2c->dev;
	tps6507x->i2c_client = i2c;
	tps6507x->read_dev = tps6507x_i2c_read_device;
	tps6507x->write_dev = tps6507x_i2c_write_device;

	ret = mfd_add_devices(tps6507x->dev, -1,
			      tps6507x_devs, ARRAY_SIZE(tps6507x_devs),
			      NULL, 0);

	if (ret < 0)
		goto err;

// read "ppath" - test chip
	ret = tps6507x->read_dev(tps6507x, 0x01, 1, &irq_mask);
	if (ret != 0) {
		printk( "TPS6507x: PMIC not responding\n");
		goto err;
	}

	if (i2c->irq > 0) {

		INIT_DELAYED_WORK(&tps6507x->isr_work, tps6507x_isr_work_func);

		// Disable all interrupts and read interrupt register
		irq_mask = 0x00;
		ret = tps6507x->write_dev(tps6507x, TPS6507X_REG_INT, 1, &irq_mask);
		if (ret == 0)
			ret = tps6507x->read_dev(tps6507x, TPS6507X_REG_INT, 1, &irq_mask);
		if (ret != 0) {
			printk( "TPS6507x: error resetting irq\n");
			goto err;
		}

		ret = request_irq(i2c->irq, tps6507x_isr,
				IRQF_TRIGGER_FALLING,
				"tps6507x", i2c);

		if (ret) {
			printk( "TPS6507x: Unable to claim irq %d; error %d\n",
				i2c->irq, ret);
			goto err;
		}

		disable_irq(i2c->irq);

		// write IRQ register to enable power button IRQ
		irq_mask = 0x20;
		ret = tps6507x->write_dev(tps6507x, TPS6507X_REG_INT, 1, &irq_mask);
		if (ret != 0) {
			printk( "TPS6507x: error writing irq mask \n");
			goto err2;
		}

		// Read IRQ register to clear pending interrupts
		ret = tps6507x->read_dev(tps6507x, TPS6507X_REG_INT, 1, &irq_mask);
		if (ret != 0) {
			printk( "TPS6507x: error reading irq mask \n");
			goto err2;
		}

		enable_irq(i2c->irq);
		device_init_wakeup(tps6507x->dev, 1);

		printk( "TPS6507x: IRQ initialized and enabled (%i, 0x%02x)\n", i2c->irq, irq_mask);

		if (tps6507x->pbtn_gpio >= 0) {
			if ( gpio_request(tps6507x->pbtn_gpio, "pb_out") < 0)
				printk(KERN_ERR "TPS6507x: failed to get PB_OUT (%i) gpio\n",tps6507x->pbtn_gpio);
		}

	}

	return ret;

err2:
	if (i2c->irq > 0)
		free_irq(i2c->irq, i2c);
err:
	mfd_remove_devices(tps6507x->dev);
	kfree(tps6507x);
	return ret;
}

static int tps6507x_i2c_remove(struct i2c_client *i2c)
{
	struct tps6507x_dev *tps6507x = i2c_get_clientdata(i2c);

	if (i2c->irq > 0)
		free_irq(i2c->irq, i2c);

	cancel_delayed_work_sync(&tps6507x->isr_work);

	mfd_remove_devices(tps6507x->dev);
	kfree(tps6507x);

	return 0;
}

#ifdef CONFIG_PM
static int tps6507x_i2c_suspend(struct i2c_client *i2c, pm_message_t state)
{
	struct tps6507x_dev *tps6507x = i2c_get_clientdata(i2c);
	u8 irq_mask;
	int ret;

	if (wakeup_by_tsc) {
		printk( "TPS6507x: waiting for TSC suspended...\n");
		while (!tps6507x->tsc_suspended)
			msleep(10);
	}

	if (device_may_wakeup(tps6507x->dev) && i2c->irq > 0) {
		tps6507x->skip_pbtn = 1;

		// Read IRQ register to clear pending interrupts
		ret = tps6507x->read_dev(tps6507x, TPS6507X_REG_INT, 1, &irq_mask);
		if (ret != 0) {
			printk( "TPS6507x: susp: error reading irq mask \n");
		}

		irq_mask = 0;
		if (wakeup_by_power_button) irq_mask |= 0x20;
		if (wakeup_by_tsc) irq_mask |= 0x40;

		// Write IRQ register - new mask
		ret = tps6507x->write_dev(tps6507x, TPS6507X_REG_INT, 1, &irq_mask);
		if (ret != 0) {
			printk( "TPS6507x: susp: error writing irq mask \n");
		}

		enable_irq_wake(i2c->irq);
		printk( "TPS6507x: wakeup by IRQ enabled\n");
	}

	printk( "TPS6507x: suspended\n");

	return 0;
}

static int tps6507x_i2c_resume(struct i2c_client *i2c)
{
	struct tps6507x_dev *tps6507x = i2c_get_clientdata(i2c);

	if (i2c->irq > 0 && device_may_wakeup(tps6507x->dev))
		disable_irq_wake(i2c->irq);

	return 0;
}
#endif

static const struct i2c_device_id tps6507x_i2c_id[] = {
       { "tps6507x", 0 },
       { }
};

MODULE_DEVICE_TABLE(i2c, tps6507x_i2c_id);


static struct i2c_driver tps6507x_i2c_driver = {
	.driver = {
		   .name = "tps6507x",
		   .owner = THIS_MODULE,
	},
	.probe = tps6507x_i2c_probe,
	.remove = tps6507x_i2c_remove,
#ifdef CONFIG_PM
	.suspend = tps6507x_i2c_suspend,
	.resume =  tps6507x_i2c_resume,
#endif
	.id_table = tps6507x_i2c_id,
};

static int __init tps6507x_i2c_init(void)
{
	return i2c_add_driver(&tps6507x_i2c_driver);
}
/* init early so consumer devices can complete system boot */
subsys_initcall(tps6507x_i2c_init);

static void __exit tps6507x_i2c_exit(void)
{
	i2c_del_driver(&tps6507x_i2c_driver);
}
module_exit(tps6507x_i2c_exit);

MODULE_DESCRIPTION("TPS6507x chip family multi-function driver");
MODULE_LICENSE("GPL");
