/*
 * Driver for keys on I2C IO expander
 *
 * Copyright 2010 Sriramakrishnan.A.G.
 *
 * Implementation is based gpio_keys.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/slab.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/gpio.h>
#include <linux/tca8418_keypad.h>
#include <linux/workqueue.h>
#include <linux/types.h>
#include <linux/interrupt.h>
#include <linux/delay.h>

static const struct i2c_device_id tca8418_id[] = {
	{ "tca8418-keys", 0, },
	{ }
};
MODULE_DEVICE_TABLE(i2c, tca8418_id);

struct tca8418_button_data {
	struct tca8418_key_codes *recode;
	struct input_dev *input;
};

struct tca8418_keypad_chip {
	uint16_t reg_output;
	uint16_t reg_direction;
	uint16_t reg_input;

	struct i2c_client *client;
	struct tca8418_button_data  *btn_data;
	struct delayed_work dwork;
	int only_recoded;
};

static int tca8418_write_reg(struct tca8418_keypad_chip *chip, int reg,
				uint8_t val)
{
	int ret;

	ret = i2c_smbus_write_byte_data(chip->client, reg, val);

	if (ret < 0) {
		dev_err(&chip->client->dev, "failed writing register\n");
		return ret;
	}

	return 0;
}

static int tca8418_read_reg(struct tca8418_keypad_chip *chip, int reg, uint8_t *val)
{
	int ret;

	ret = i2c_smbus_read_byte_data(chip->client, reg);

	if (ret < 0) {
		dev_err(&chip->client->dev, "failed reading register\n");
		return ret;
	}

	*val = (uint8_t)ret;
	return 0;
}

static irqreturn_t tca8418_keys_isr(int irq, void *dev_id)
{
	struct i2c_client *client = dev_id;
	struct tca8418_keypad_chip * chip = i2c_get_clientdata(client);

	disable_irq_nosync(irq);
	schedule_delayed_work(&chip->dwork,0);
	return IRQ_HANDLED;
}

static void tca8418_keys_work_func(struct work_struct *workstruct)
{
	struct delayed_work *delay_work =
		container_of(workstruct, struct delayed_work, work);
	struct tca8418_keypad_chip *chip =
		container_of(delay_work, struct tca8418_keypad_chip, dwork);
	struct tca8418_button_data *bdata = chip->btn_data;
	struct input_dev *input = bdata->input;
	struct tca8418_key_codes *recode = bdata->recode;
	uint8_t reg_val;
	int ret, state;

	while (1) {
		reg_val = 0;
		ret = tca8418_read_reg(chip, TCA8418_REG_INT_STAT, &reg_val);
		// If no keyboard interrupt pending, exit loop
		if ( (reg_val & 1) == 0 ) break;
		while (1) {
			reg_val = 0;
			ret = tca8418_read_reg(chip, TCA8418_REG_KEY_LCK_EC, &reg_val);
			// if no keys in FIFO - exit loop
			if ((reg_val & 0x0F) == 0) break;
			// Read scan code from FIFO
			ret = tca8418_read_reg(chip, TCA8418_REG_KEY_EVENT_A, &reg_val);
			// Recode from scan code to key code
			state = (reg_val & 0x80) != 0;
			reg_val &= 0x7F;
			if (recode) {
				while (recode->scancode >= 0 && recode->scancode != reg_val) recode++;
				if (recode->scancode >= 0) {
				    input_event(input, EV_KEY, recode->keycode, state);
				    input_sync(input);
				} else if (!chip->only_recoded) {
				    input_event(input, EV_KEY, reg_val, state);
				    input_sync(input);
				}
			} else if (!chip->only_recoded) {
			    input_event(input, EV_KEY, reg_val, state);
			    input_sync(input);
			}
		}
		// clear pending interrupts
		ret = tca8418_write_reg(chip, TCA8418_REG_INT_STAT, 0x1F);
	}

	if(chip->client->irq <= 0)
		schedule_delayed_work(&chip->dwork, msecs_to_jiffies(100));
	else
		enable_irq(chip->client->irq);

}


static int __devinit tca8418_keypad_probe(struct i2c_client *client,
				   const struct i2c_device_id *id)
{
	struct tca8418_keys_platform_data *pdata;
	struct tca8418_keypad_chip *chip;
	struct tca8418_button_data *bdata;
	struct tca8418_key_codes *recode = NULL, * rec;
	struct input_dev *input;
	int i, ret, code;
	uint8_t reg_val;

	chip = kzalloc(sizeof(struct tca8418_keypad_chip), GFP_KERNEL);
	if (chip == NULL)
		return -ENOMEM;

	pdata = client->dev.platform_data;
	if (pdata == NULL) {
		dev_dbg(&client->dev, "no platform data\n");
		ret = -EINVAL;
		goto fail1;
	}

	chip->client = client;

	// zero CFG register
	ret = tca8418_write_reg(chip, TCA8418_REG_CFG, 0);
	if (ret)
		goto fail1;

	// set all lines to GPIO
	
	ret = tca8418_write_reg(chip, TCA8418_REG_KP_GPIO1, 0x00);
	if (ret)
		goto fail1;
	ret = tca8418_write_reg(chip, TCA8418_REG_KP_GPIO2, 0x00);
	if (ret)
		goto fail1;
	ret = tca8418_write_reg(chip, TCA8418_REG_KP_GPIO3, 0x00);
	if (ret)
		goto fail1;

// Zero all INTs, EMs
	
	ret = tca8418_write_reg(chip, TCA8418_REG_GPIO_INT_EN1, 0x00);
	if (ret)
		goto fail1;
	ret = tca8418_write_reg(chip, TCA8418_REG_GPIO_INT_EN2, 0x00);
	if (ret)
		goto fail1;
	ret = tca8418_write_reg(chip, TCA8418_REG_GPIO_INT_EN3, 0x00);
	if (ret)
		goto fail1;
	
	ret = tca8418_write_reg(chip, TCA8418_REG_GPI_EM1, 0x00);
	if (ret)
		goto fail1;
	ret = tca8418_write_reg(chip, TCA8418_REG_GPI_EM2, 0x00);
	if (ret)
		goto fail1;
	ret = tca8418_write_reg(chip, TCA8418_REG_GPI_EM3, 0x00);
	if (ret)
		goto fail1;

// Clear all GPIO_INT_STAT
	ret = tca8418_read_reg(chip, TCA8418_REG_GPIO_INT_STAT1, &reg_val);
	if (ret)
		goto fail1;
	ret = tca8418_read_reg(chip, TCA8418_REG_GPIO_INT_STAT2, &reg_val);
	if (ret)
		goto fail1;
	ret = tca8418_read_reg(chip, TCA8418_REG_GPIO_INT_STAT3, &reg_val);
	if (ret)
		goto fail1;

// Read all pending events from FIFO
	while (1) {
		reg_val = 0;
		ret = tca8418_read_reg(chip, TCA8418_REG_KEY_LCK_EC, &reg_val);
		if (ret)
			goto fail1;
		// if no keys in FIFO - exit loop
		if ((reg_val & 0x0F) == 0) break;
		// Read scan code from FIFO
		ret = tca8418_read_reg(chip, TCA8418_REG_KEY_EVENT_A, &reg_val);
		if (ret)
			goto fail1;
	}

	// clear pending interrupts
	ret = tca8418_write_reg(chip, TCA8418_REG_INT_STAT, 0x1F);
	if (ret)
		goto fail1;

	// set all lines to keypad matrix
	
	ret = tca8418_write_reg(chip, TCA8418_REG_KP_GPIO1, 0xFF);
	if (ret)
		goto fail1;
	ret = tca8418_write_reg(chip, TCA8418_REG_KP_GPIO2, 0xFF);
	if (ret)
		goto fail1;
	ret = tca8418_write_reg(chip, TCA8418_REG_KP_GPIO3, 0x03);
	if (ret)
		goto fail1;

	// Enable key interrupt
	ret = tca8418_write_reg(chip, TCA8418_REG_CFG, 1);
	if (ret)
		goto fail1;

	i2c_set_clientdata(client, chip);

	// calculate size of recode table and create copy of it
	i=0; recode = pdata->recode;
	if (recode) {
		i++;
		while (recode->scancode >= 0) {recode++; i++; }
		recode = kzalloc(sizeof(struct tca8418_key_codes) * i, GFP_KERNEL);
		if (!recode) {
			ret = -ENOMEM;
			goto fail1;
		}
		memcpy(recode, pdata->recode, sizeof(struct tca8418_key_codes) * i);
	}

	bdata = kzalloc(sizeof(struct tca8418_button_data), GFP_KERNEL);
	if(!bdata) {
		ret = -ENOMEM;
		goto fail1;
	}
	
	bdata->recode = recode;

	input = input_allocate_device();
	if (!input) {
		printk("failed to allocate state\n");
		ret = -ENOMEM;
		goto fail2;
	}

	input->phys = "tca8418-keys/input0";
	input->name = "tca8418-keypad";
	input->dev.parent = &client->dev;

	input->id.bustype = BUS_HOST;
	input->id.vendor = 0x0001;
	input->id.product = 0x0001;
	input->id.version = 0x0100;

	/* Enable auto repeat feature of Linux input subsystem */
	if (pdata->rep)
		__set_bit(EV_REP, input->evbit);

	chip->btn_data = bdata;
	chip->only_recoded = pdata->only_recoded;
	bdata->input = input;

	for (i = 0; i < 80; i++) {
		rec = recode; code = i;
		if (rec) {
			while (rec->scancode >= 0 && rec->scancode != code) rec++;
			if (rec->scancode >= 0) {
				code = rec->keycode;
				if (chip->only_recoded) input_set_capability(input, EV_KEY, code);
			}
		}
		if (!chip->only_recoded) input_set_capability(input, EV_KEY, code);
	}

	INIT_DELAYED_WORK(&chip->dwork, tca8418_keys_work_func);

	if (client->irq > 0) {
		ret = request_irq(client->irq, tca8418_keys_isr,
				IRQF_TRIGGER_LOW,
				"tca8418-keypad", client);
		if (ret) {
			printk( "Unable to claim irq %d; error %d\n",
				client->irq, ret);
			goto fail3;
		}
		disable_irq(client->irq);
	}

	ret = input_register_device(input);
	if (ret) {
		printk( "Unable to register input device, "
			"error: %d\n", ret);
		goto fail3;
	}

	if(client->irq <= 0)
		schedule_delayed_work(&chip->dwork, msecs_to_jiffies(100));
	else {
		enable_irq(client->irq);
		printk("TCA8418: enable_irq %d\n", client->irq);
	}

	dev_info(&chip->client->dev, " TCA8418 based keypad\n");

	return 0;
fail3:
	input_free_device(input);
fail2:
	kfree(bdata);
fail1:
	if (recode) kfree(recode);
	kfree(chip);
	return ret;
}

static int tca8418_keypad_remove(struct i2c_client *client)
{
	struct tca8418_keypad_chip *chip = i2c_get_clientdata(client);
	struct tca8418_button_data *bdata = chip->btn_data;
	struct input_dev *input = bdata->input;

	if(client->irq > 0)
		free_irq(client->irq,chip);
	cancel_delayed_work_sync(&chip->dwork);
	input_unregister_device(input);
	input_free_device(input);
	if (bdata->recode) kfree(bdata->recode);
	kfree(bdata);
	kfree(chip);
	return 0;
}

static struct i2c_driver tca8418_keypad_driver = {
	.driver = {
		.name	= "tca8418-keypad",
	},
	.probe		= tca8418_keypad_probe,
	.remove		= tca8418_keypad_remove,
	.id_table	= tca8418_id,
};

static int __init tca8418_keypad_init(void)
{
	return i2c_add_driver(&tca8418_keypad_driver);
}

subsys_initcall(tca8418_keypad_init);

static void __exit tca8418_keypad_exit(void)
{
	i2c_del_driver(&tca8418_keypad_driver);
}
module_exit(tca8418_keypad_exit);

MODULE_AUTHOR("Sergey Markov <sm@venus.ru>");
MODULE_DESCRIPTION("Keypad driver for tca8418 keyboard/GPIO controller");
MODULE_LICENSE("GPL");
