/**
 * Implements support for TPS6507x
 *
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation version 2.
 *
 * This program is distributed "as is" WITHOUT ANY WARRANTY of any
 * kind, whether express or implied; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/err.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/i2c/twl.h>

#include <plat/voltage.h>

#include "pm.h"

/* Supported voltage values for regulators (in milliVolts) */
static const u16 VDCDCx_VSEL_table[] = {
	725, 750, 775, 800,
	825, 850, 875, 900,
	925, 950, 975, 1000,
	1025, 1050, 1075, 1100,
	1125, 1150, 1175, 1200,
	1225, 1250, 1275, 1300,
	1325, 1350, 1375, 1400,
	1425, 1450, 1475, 1500,
	1550, 1600, 1650, 1700,
	1750, 1800, 1850, 1900,
	1950, 2000, 2050, 2100,
	2150, 2200, 2250, 2300,
	2350, 2400, 2450, 2500,
	2550, 2600, 2650, 2700,
	2750, 2800, 2850, 2900,
	3000, 3100, 3200, 3300,
};

/*
 * Get voltage corresponding to specified vsel value using this formula:
 *	Vout = 0.8V + (25mV x Vsel)
 */
static unsigned long tps6507x_dcdc3_vsel_to_uv(const u8 vsel)
{
	if (vsel > 63)
		return 3300000;
	else
		return VDCDCx_VSEL_table[vsel];
}

/*
 * Get vsel value corresponding to specified voltage using this formula:
 *	Vsel = (Vout - 0.8V)/ 25mV
 */
static u8 tps6507x_dcdc3_uv_to_vsel(unsigned long uv)
{
	int i;

	uv /= 1000;
	for (i=0; i<64; i++) {
		if (VDCDCx_VSEL_table[i] >= uv)
			return i;
	}

	return 63;
}

/*
 * Voltage information related to the MPU voltage domain of the
 * AM35x processors - in relation to the TPS6507x.
 */

static struct omap_volt_pmic_info tps6507x_am35xx_mpu_volt_info = {
	.on_volt                = 1200000,
	.vsel_to_uv		= tps6507x_dcdc3_vsel_to_uv,
	.uv_to_vsel		= tps6507x_dcdc3_uv_to_vsel,

};


int __init omap3_tps6507x_init(void)
{
	struct voltagedomain *voltdm;

	if (!cpu_is_omap34xx())
		return -ENODEV;

	if (cpu_is_omap3505() || cpu_is_omap3517()) {
		voltdm = omap_voltage_domain_lookup("mpu");
		omap_voltage_register_pmic(voltdm, &tps6507x_am35xx_mpu_volt_info);
		voltdm = omap_voltage_domain_lookup("core");
		omap_voltage_register_pmic(voltdm, &tps6507x_am35xx_mpu_volt_info);
	}

	return 0;
}
